import { Config, SearchResponse, SearchParams, IndexObject, IndexOptions, IndexStats, DocumentsQuery, DocumentQuery, DocumentOptions, Settings, Synonyms, StopWords, RankingRules, DistinctAttribute, FilterableAttributes, SortableAttributes, SearchableAttributes, DisplayedAttributes, TypoTolerance, WaitOptions, TasksQuery, TasksResults, PaginationSettings, Faceting, ResourceResults, RawDocumentAdditionOptions, ContentType, DocumentsIds, DocumentsDeletionQuery, SearchForFacetValuesParams, SearchForFacetValuesResponse, SeparatorTokens, NonSeparatorTokens, Dictionary, ProximityPrecision, Embedders, SearchCutoffMs } from './types';
import { HttpRequests } from './http-requests';
import { Task, TaskClient } from './task';
import { EnqueuedTask } from './enqueued-task';
declare class Index<T extends Record<string, any> = Record<string, any>> {
    uid: string;
    primaryKey: string | undefined;
    createdAt: Date | undefined;
    updatedAt: Date | undefined;
    httpRequest: HttpRequests;
    tasks: TaskClient;
    /**
     * @param config - Request configuration options
     * @param uid - UID of the index
     * @param primaryKey - Primary Key of the index
     */
    constructor(config: Config, uid: string, primaryKey?: string);
    /**
     * Search for documents into an index
     *
     * @param query - Query string
     * @param options - Search options
     * @param config - Additional request configuration options
     * @returns Promise containing the search response
     */
    search<D extends Record<string, any> = T, S extends SearchParams = SearchParams>(query?: string | null, options?: S, config?: Partial<Request>): Promise<SearchResponse<D, S>>;
    /**
     * Search for documents into an index using the GET method
     *
     * @param query - Query string
     * @param options - Search options
     * @param config - Additional request configuration options
     * @returns Promise containing the search response
     */
    searchGet<D extends Record<string, any> = T, S extends SearchParams = SearchParams>(query?: string | null, options?: S, config?: Partial<Request>): Promise<SearchResponse<D, S>>;
    /**
     * Search for facet values
     *
     * @param params - Parameters used to search on the facets
     * @param config - Additional request configuration options
     * @returns Promise containing the search response
     */
    searchForFacetValues(params: SearchForFacetValuesParams, config?: Partial<Request>): Promise<SearchForFacetValuesResponse>;
    /**
     * Get index information.
     *
     * @returns Promise containing index information
     */
    getRawInfo(): Promise<IndexObject>;
    /**
     * Fetch and update Index information.
     *
     * @returns Promise to the current Index object with updated information
     */
    fetchInfo(): Promise<this>;
    /**
     * Get Primary Key.
     *
     * @returns Promise containing the Primary Key of the index
     */
    fetchPrimaryKey(): Promise<string | undefined>;
    /**
     * Create an index.
     *
     * @param uid - Unique identifier of the Index
     * @param options - Index options
     * @param config - Request configuration options
     * @returns Newly created Index object
     */
    static create(uid: string, options: IndexOptions | undefined, config: Config): Promise<EnqueuedTask>;
    /**
     * Update an index.
     *
     * @param data - Data to update
     * @returns Promise to the current Index object with updated information
     */
    update(data: IndexOptions): Promise<EnqueuedTask>;
    /**
     * Delete an index.
     *
     * @returns Promise which resolves when index is deleted successfully
     */
    delete(): Promise<EnqueuedTask>;
    /**
     * Get the list of all the tasks of the index.
     *
     * @param parameters - Parameters to browse the tasks
     * @returns Promise containing all tasks
     */
    getTasks(parameters?: TasksQuery): Promise<TasksResults>;
    /**
     * Get one task of the index.
     *
     * @param taskUid - Task identifier
     * @returns Promise containing a task
     */
    getTask(taskUid: number): Promise<Task>;
    /**
     * Wait for multiple tasks to be processed.
     *
     * @param taskUids - Tasks identifier
     * @param waitOptions - Options on timeout and interval
     * @returns Promise containing an array of tasks
     */
    waitForTasks(taskUids: number[], { timeOutMs, intervalMs }?: WaitOptions): Promise<Task[]>;
    /**
     * Wait for a task to be processed.
     *
     * @param taskUid - Task identifier
     * @param waitOptions - Options on timeout and interval
     * @returns Promise containing an array of tasks
     */
    waitForTask(taskUid: number, { timeOutMs, intervalMs }?: WaitOptions): Promise<Task>;
    /**
     * Get stats of an index
     *
     * @returns Promise containing object with stats of the index
     */
    getStats(): Promise<IndexStats>;
    /**
     * Get documents of an index.
     *
     * @param parameters - Parameters to browse the documents. Parameters can
     *   contain the `filter` field only available in Meilisearch v1.2 and newer
     * @returns Promise containing the returned documents
     */
    getDocuments<D extends Record<string, any> = T>(parameters?: DocumentsQuery<D>): Promise<ResourceResults<D[]>>;
    /**
     * Get one document
     *
     * @param documentId - Document ID
     * @param parameters - Parameters applied on a document
     * @returns Promise containing Document response
     */
    getDocument<D extends Record<string, any> = T>(documentId: string | number, parameters?: DocumentQuery<T>): Promise<D>;
    /**
     * Add or replace multiples documents to an index
     *
     * @param documents - Array of Document objects to add/replace
     * @param options - Options on document addition
     * @returns Promise containing an EnqueuedTask
     */
    addDocuments(documents: T[], options?: DocumentOptions): Promise<EnqueuedTask>;
    /**
     * Add or replace multiples documents in a string format to an index. It only
     * supports csv, ndjson and json formats.
     *
     * @param documents - Documents provided in a string to add/replace
     * @param contentType - Content type of your document:
     *   'text/csv'|'application/x-ndjson'|'application/json'
     * @param options - Options on document addition
     * @returns Promise containing an EnqueuedTask
     */
    addDocumentsFromString(documents: string, contentType: ContentType, queryParams?: RawDocumentAdditionOptions): Promise<EnqueuedTask>;
    /**
     * Add or replace multiples documents to an index in batches
     *
     * @param documents - Array of Document objects to add/replace
     * @param batchSize - Size of the batch
     * @param options - Options on document addition
     * @returns Promise containing array of enqueued task objects for each batch
     */
    addDocumentsInBatches(documents: T[], batchSize?: number, options?: DocumentOptions): Promise<EnqueuedTask[]>;
    /**
     * Add or update multiples documents to an index
     *
     * @param documents - Array of Document objects to add/update
     * @param options - Options on document update
     * @returns Promise containing an EnqueuedTask
     */
    updateDocuments(documents: Array<Partial<T>>, options?: DocumentOptions): Promise<EnqueuedTask>;
    /**
     * Add or update multiples documents to an index in batches
     *
     * @param documents - Array of Document objects to add/update
     * @param batchSize - Size of the batch
     * @param options - Options on document update
     * @returns Promise containing array of enqueued task objects for each batch
     */
    updateDocumentsInBatches(documents: Array<Partial<T>>, batchSize?: number, options?: DocumentOptions): Promise<EnqueuedTask[]>;
    /**
     * Add or update multiples documents in a string format to an index. It only
     * supports csv, ndjson and json formats.
     *
     * @param documents - Documents provided in a string to add/update
     * @param contentType - Content type of your document:
     *   'text/csv'|'application/x-ndjson'|'application/json'
     * @param queryParams - Options on raw document addition
     * @returns Promise containing an EnqueuedTask
     */
    updateDocumentsFromString(documents: string, contentType: ContentType, queryParams?: RawDocumentAdditionOptions): Promise<EnqueuedTask>;
    /**
     * Delete one document
     *
     * @param documentId - Id of Document to delete
     * @returns Promise containing an EnqueuedTask
     */
    deleteDocument(documentId: string | number): Promise<EnqueuedTask>;
    /**
     * Delete multiples documents of an index.
     *
     * @param params - Params value can be:
     *
     *   - DocumentsDeletionQuery: An object containing the parameters to customize
     *       your document deletion. Only available in Meilisearch v1.2 and newer
     *   - DocumentsIds: An array of document ids to delete
     *
     * @returns Promise containing an EnqueuedTask
     */
    deleteDocuments(params: DocumentsDeletionQuery | DocumentsIds): Promise<EnqueuedTask>;
    /**
     * Delete all documents of an index
     *
     * @returns Promise containing an EnqueuedTask
     */
    deleteAllDocuments(): Promise<EnqueuedTask>;
    /**
     * Retrieve all settings
     *
     * @returns Promise containing Settings object
     */
    getSettings(): Promise<Settings>;
    /**
     * Update all settings Any parameters not provided will be left unchanged.
     *
     * @param settings - Object containing parameters with their updated values
     * @returns Promise containing an EnqueuedTask
     */
    updateSettings(settings: Settings): Promise<EnqueuedTask>;
    /**
     * Reset settings.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetSettings(): Promise<EnqueuedTask>;
    /**
     * Get the pagination settings.
     *
     * @returns Promise containing object of pagination settings
     */
    getPagination(): Promise<PaginationSettings>;
    /**
     * Update the pagination settings.
     *
     * @param pagination - Pagination object
     * @returns Promise containing an EnqueuedTask
     */
    updatePagination(pagination: PaginationSettings): Promise<EnqueuedTask>;
    /**
     * Reset the pagination settings.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetPagination(): Promise<EnqueuedTask>;
    /**
     * Get the list of all synonyms
     *
     * @returns Promise containing object of synonym mappings
     */
    getSynonyms(): Promise<object>;
    /**
     * Update the list of synonyms. Overwrite the old list.
     *
     * @param synonyms - Mapping of synonyms with their associated words
     * @returns Promise containing an EnqueuedTask
     */
    updateSynonyms(synonyms: Synonyms): Promise<EnqueuedTask>;
    /**
     * Reset the synonym list to be empty again
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetSynonyms(): Promise<EnqueuedTask>;
    /**
     * Get the list of all stop-words
     *
     * @returns Promise containing array of stop-words
     */
    getStopWords(): Promise<string[]>;
    /**
     * Update the list of stop-words. Overwrite the old list.
     *
     * @param stopWords - Array of strings that contains the stop-words.
     * @returns Promise containing an EnqueuedTask
     */
    updateStopWords(stopWords: StopWords): Promise<EnqueuedTask>;
    /**
     * Reset the stop-words list to be empty again
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetStopWords(): Promise<EnqueuedTask>;
    /**
     * Get the list of all ranking-rules
     *
     * @returns Promise containing array of ranking-rules
     */
    getRankingRules(): Promise<string[]>;
    /**
     * Update the list of ranking-rules. Overwrite the old list.
     *
     * @param rankingRules - Array that contain ranking rules sorted by order of
     *   importance.
     * @returns Promise containing an EnqueuedTask
     */
    updateRankingRules(rankingRules: RankingRules): Promise<EnqueuedTask>;
    /**
     * Reset the ranking rules list to its default value
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetRankingRules(): Promise<EnqueuedTask>;
    /**
     * Get the distinct-attribute
     *
     * @returns Promise containing the distinct-attribute of the index
     */
    getDistinctAttribute(): Promise<string | null>;
    /**
     * Update the distinct-attribute.
     *
     * @param distinctAttribute - Field name of the distinct-attribute
     * @returns Promise containing an EnqueuedTask
     */
    updateDistinctAttribute(distinctAttribute: DistinctAttribute): Promise<EnqueuedTask>;
    /**
     * Reset the distinct-attribute.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetDistinctAttribute(): Promise<EnqueuedTask>;
    /**
     * Get the filterable-attributes
     *
     * @returns Promise containing an array of filterable-attributes
     */
    getFilterableAttributes(): Promise<string[]>;
    /**
     * Update the filterable-attributes.
     *
     * @param filterableAttributes - Array of strings containing the attributes
     *   that can be used as filters at query time
     * @returns Promise containing an EnqueuedTask
     */
    updateFilterableAttributes(filterableAttributes: FilterableAttributes): Promise<EnqueuedTask>;
    /**
     * Reset the filterable-attributes.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetFilterableAttributes(): Promise<EnqueuedTask>;
    /**
     * Get the sortable-attributes
     *
     * @returns Promise containing array of sortable-attributes
     */
    getSortableAttributes(): Promise<string[]>;
    /**
     * Update the sortable-attributes.
     *
     * @param sortableAttributes - Array of strings containing the attributes that
     *   can be used to sort search results at query time
     * @returns Promise containing an EnqueuedTask
     */
    updateSortableAttributes(sortableAttributes: SortableAttributes): Promise<EnqueuedTask>;
    /**
     * Reset the sortable-attributes.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetSortableAttributes(): Promise<EnqueuedTask>;
    /**
     * Get the searchable-attributes
     *
     * @returns Promise containing array of searchable-attributes
     */
    getSearchableAttributes(): Promise<string[]>;
    /**
     * Update the searchable-attributes.
     *
     * @param searchableAttributes - Array of strings that contains searchable
     *   attributes sorted by order of importance(most to least important)
     * @returns Promise containing an EnqueuedTask
     */
    updateSearchableAttributes(searchableAttributes: SearchableAttributes): Promise<EnqueuedTask>;
    /**
     * Reset the searchable-attributes.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetSearchableAttributes(): Promise<EnqueuedTask>;
    /**
     * Get the displayed-attributes
     *
     * @returns Promise containing array of displayed-attributes
     */
    getDisplayedAttributes(): Promise<string[]>;
    /**
     * Update the displayed-attributes.
     *
     * @param displayedAttributes - Array of strings that contains attributes of
     *   an index to display
     * @returns Promise containing an EnqueuedTask
     */
    updateDisplayedAttributes(displayedAttributes: DisplayedAttributes): Promise<EnqueuedTask>;
    /**
     * Reset the displayed-attributes.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetDisplayedAttributes(): Promise<EnqueuedTask>;
    /**
     * Get the typo tolerance settings.
     *
     * @returns Promise containing the typo tolerance settings.
     */
    getTypoTolerance(): Promise<TypoTolerance>;
    /**
     * Update the typo tolerance settings.
     *
     * @param typoTolerance - Object containing the custom typo tolerance
     *   settings.
     * @returns Promise containing object of the enqueued update
     */
    updateTypoTolerance(typoTolerance: TypoTolerance): Promise<EnqueuedTask>;
    /**
     * Reset the typo tolerance settings.
     *
     * @returns Promise containing object of the enqueued update
     */
    resetTypoTolerance(): Promise<EnqueuedTask>;
    /**
     * Get the faceting settings.
     *
     * @returns Promise containing object of faceting index settings
     */
    getFaceting(): Promise<Faceting>;
    /**
     * Update the faceting settings.
     *
     * @param faceting - Faceting index settings object
     * @returns Promise containing an EnqueuedTask
     */
    updateFaceting(faceting: Faceting): Promise<EnqueuedTask>;
    /**
     * Reset the faceting settings.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetFaceting(): Promise<EnqueuedTask>;
    /**
     * Get the list of all separator tokens.
     *
     * @returns Promise containing array of separator tokens
     */
    getSeparatorTokens(): Promise<string[]>;
    /**
     * Update the list of separator tokens. Overwrite the old list.
     *
     * @param separatorTokens - Array that contains separator tokens.
     * @returns Promise containing an EnqueuedTask or null
     */
    updateSeparatorTokens(separatorTokens: SeparatorTokens): Promise<EnqueuedTask>;
    /**
     * Reset the separator tokens list to its default value
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetSeparatorTokens(): Promise<EnqueuedTask>;
    /**
     * Get the list of all non-separator tokens.
     *
     * @returns Promise containing array of non-separator tokens
     */
    getNonSeparatorTokens(): Promise<string[]>;
    /**
     * Update the list of non-separator tokens. Overwrite the old list.
     *
     * @param nonSeparatorTokens - Array that contains non-separator tokens.
     * @returns Promise containing an EnqueuedTask or null
     */
    updateNonSeparatorTokens(nonSeparatorTokens: NonSeparatorTokens): Promise<EnqueuedTask>;
    /**
     * Reset the non-separator tokens list to its default value
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetNonSeparatorTokens(): Promise<EnqueuedTask>;
    /**
     * Get the dictionary settings of a Meilisearch index.
     *
     * @returns Promise containing the dictionary settings
     */
    getDictionary(): Promise<string[]>;
    /**
     * Update the dictionary settings. Overwrite the old settings.
     *
     * @param dictionary - Array that contains the new dictionary settings.
     * @returns Promise containing an EnqueuedTask or null
     */
    updateDictionary(dictionary: Dictionary): Promise<EnqueuedTask>;
    /**
     * Reset the dictionary settings to its default value
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetDictionary(): Promise<EnqueuedTask>;
    /**
     * Get the proximity precision settings of a Meilisearch index.
     *
     * @returns Promise containing the proximity precision settings
     */
    getProximityPrecision(): Promise<ProximityPrecision>;
    /**
     * Update the proximity precision settings. Overwrite the old settings.
     *
     * @param proximityPrecision - String that contains the new proximity
     *   precision settings.
     * @returns Promise containing an EnqueuedTask or null
     */
    updateProximityPrecision(proximityPrecision: ProximityPrecision): Promise<EnqueuedTask>;
    /**
     * Reset the proximity precision settings to its default value
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetProximityPrecision(): Promise<EnqueuedTask>;
    /**
     * Get the embedders settings of a Meilisearch index.
     *
     * @returns Promise containing the embedders settings
     */
    getEmbedders(): Promise<Embedders>;
    /**
     * Update the embedders settings. Overwrite the old settings.
     *
     * @param embedders - Object that contains the new embedders settings.
     * @returns Promise containing an EnqueuedTask or null
     */
    updateEmbedders(embedders: Embedders): Promise<EnqueuedTask>;
    /**
     * Reset the embedders settings to its default value
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetEmbedders(): Promise<EnqueuedTask>;
    /**
     * Get the SearchCutoffMs settings.
     *
     * @returns Promise containing object of SearchCutoffMs settings
     */
    getSearchCutoffMs(): Promise<SearchCutoffMs>;
    /**
     * Update the SearchCutoffMs settings.
     *
     * @param searchCutoffMs - Object containing SearchCutoffMsSettings
     * @returns Promise containing an EnqueuedTask
     */
    updateSearchCutoffMs(searchCutoffMs: SearchCutoffMs): Promise<EnqueuedTask>;
    /**
     * Reset the SearchCutoffMs settings.
     *
     * @returns Promise containing an EnqueuedTask
     */
    resetSearchCutoffMs(): Promise<EnqueuedTask>;
}
export { Index };
//# sourceMappingURL=indexes.d.ts.map